<?php
// submit.php (with MySQL PDO integration)
header('Content-Type: application/json; charset=utf-8');

// ---------- helper ----------
function respond($ok, $msg, $extra = []) {
    echo json_encode(array_merge(['success' => $ok, 'message' => $msg], $extra));
    exit;
}

function clean($v) {
    return isset($v) ? trim(htmlspecialchars($v, ENT_QUOTES, 'UTF-8')) : null;
}

// ---------- config ----------
$uploadDir = __DIR__ . '/uploads/';
if (!is_dir($uploadDir)) {
    if (!mkdir($uploadDir, 0755, true)) {
        respond(false, 'Server error: cannot create upload directory.');
    }
}

$allowedImageTypes = ['image/jpeg','image/png','image/jpg'];
$maxFileSize = 2 * 1024 * 1024; // 2MB

// ---------- basic required validation ----------
$required = ['class_sought','child_name','dob','email','father_name','mother_name','father_phone','mother_phone','declaration'];
foreach ($required as $r) {
    if (empty($_POST[$r])) {
        respond(false, "Missing required field: $r");
    }
}
if (!filter_var($_POST['email'], FILTER_VALIDATE_EMAIL)) {
    respond(false, "Invalid email address.");
}
$fp = preg_replace('/\D/','', $_POST['father_phone']);
$mp = preg_replace('/\D/','', $_POST['mother_phone']);
if (strlen($fp) < 10 || strlen($mp) < 10) {
    respond(false, "Please provide valid 10-digit phone numbers.");
}

// ---------- file upload handling (same as before) ----------
$filesToSave = [
    'photo_child' => null,
    'photo_father' => null,
    'photo_mother' => null,
    'sign_mother' => null,
    'sign_father' => null
];

foreach ($filesToSave as $field => $v) {
    if (!empty($_FILES[$field]) && $_FILES[$field]['error'] !== UPLOAD_ERR_NO_FILE) {
        $f = $_FILES[$field];

        if ($f['error'] !== UPLOAD_ERR_OK) {
            respond(false, "Error uploading file: $field");
        }
        if ($f['size'] > $maxFileSize) {
            respond(false, "File too large for $field (max 2MB).");
        }
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mime = finfo_file($finfo, $f['tmp_name']);
        finfo_close($finfo);
        if (!in_array($mime, $allowedImageTypes)) {
            respond(false, "Invalid file type for $field. Only JPG/PNG allowed.");
        }

        $ext = pathinfo($f['name'], PATHINFO_EXTENSION);
        $safeBase = bin2hex(random_bytes(8));
        $newName = $safeBase . '_' . time() . '.' . $ext;
        $dest = $uploadDir . $newName;

        if (!move_uploaded_file($f['tmp_name'], $dest)) {
            respond(false, "Failed to save uploaded file for $field.");
        }

        // store relative path
        $filesToSave[$field] = 'uploads/' . $newName;
    }
}

// ---------- collect & sanitize POST fields ----------
$input = [];
$fields = [
    'class_sought','reg_no','child_name','gender','nationality','dob','aadhar_child','res_address','last_school','email',
    'father_name','father_aadhar','father_phone','father_edu','father_occ','father_desig','father_org','father_org_addr',
    'mother_name','mother_aadhar','mother_phone','mother_edu','mother_occ','mother_desig','mother_org','mother_org_addr',
    'single_parent','sibling_name','sibling_class','medical_need','medical_details','transfer_info','transport_req','category'
];
foreach ($fields as $f) {
    $input[$f] = isset($_POST[$f]) ? clean($_POST[$f]) : null;
}
// boolean declaration
$input['declaration'] = !empty($_POST['declaration']) ? 1 : 0;

// attach file paths
$input['photo_child_path']  = $filesToSave['photo_child'] ?? null;
$input['photo_father_path'] = $filesToSave['photo_father'] ?? null;
$input['photo_mother_path'] = $filesToSave['photo_mother'] ?? null;
$input['sign_mother_path']   = $filesToSave['sign_mother'] ?? null;
$input['sign_father_path']   = $filesToSave['sign_father'] ?? null;

// ---------- MySQL (PDO) insert ----------
$dbConfig = [
    'host' => '127.0.0.1',
    'port' => '3306',
    'dbname' => 'admission_delhi',
    'user' => 'admission_delhi_user',
    'pass' => '5AXobQM;ND0gTVwW',
    'charset' => 'utf8mb4'
];

$dsn = "mysql:host={$dbConfig['host']};port={$dbConfig['port']};dbname={$dbConfig['dbname']};charset={$dbConfig['charset']}";

try {
    $pdo = new PDO($dsn, $dbConfig['user'], $dbConfig['pass'], [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    ]);

    $pdo->beginTransaction();

    $sql = "INSERT INTO applications (
        class_sought, reg_no, child_name, gender, nationality, dob, aadhar_child, res_address, last_school, email,
        father_name, father_aadhar, father_phone, father_edu, father_occ, father_desig, father_org, father_org_addr,
        mother_name, mother_aadhar, mother_phone, mother_edu, mother_occ, mother_desig, mother_org, mother_org_addr,
        single_parent, sibling_name, sibling_class, medical_need, medical_details, transfer_info, transport_req, category,
        declaration, photo_child_path, photo_father_path, photo_mother_path, sign_mother_path, sign_father_path
    ) VALUES (
        :class_sought, :reg_no, :child_name, :gender, :nationality, :dob, :aadhar_child, :res_address, :last_school, :email,
        :father_name, :father_aadhar, :father_phone, :father_edu, :father_occ, :father_desig, :father_org, :father_org_addr,
        :mother_name, :mother_aadhar, :mother_phone, :mother_edu, :mother_occ, :mother_desig, :mother_org, :mother_org_addr,
        :single_parent, :sibling_name, :sibling_class, :medical_need, :medical_details, :transfer_info, :transport_req, :category,
        :declaration, :photo_child_path, :photo_father_path, :photo_mother_path, :sign_mother_path, :sign_father_path
    )";

    $stmt = $pdo->prepare($sql);

    // bind values
    $stmt->bindValue(':class_sought', $input['class_sought']);
    $stmt->bindValue(':reg_no', $input['reg_no']);
    $stmt->bindValue(':child_name', $input['child_name']);
    $stmt->bindValue(':gender', $input['gender']);
    $stmt->bindValue(':nationality', $input['nationality']);
    $stmt->bindValue(':dob', $input['dob']);
    $stmt->bindValue(':aadhar_child', $input['aadhar_child']);
    $stmt->bindValue(':res_address', $input['res_address']);
    $stmt->bindValue(':last_school', $input['last_school']);
    $stmt->bindValue(':email', $input['email']);

    $stmt->bindValue(':father_name', $input['father_name']);
    $stmt->bindValue(':father_aadhar', $input['father_aadhar']);
    $stmt->bindValue(':father_phone', $input['father_phone']);
    $stmt->bindValue(':father_edu', $input['father_edu']);
    $stmt->bindValue(':father_occ', $input['father_occ']);
    $stmt->bindValue(':father_desig', $input['father_desig']);
    $stmt->bindValue(':father_org', $input['father_org']);
    $stmt->bindValue(':father_org_addr', $input['father_org_addr']);

    $stmt->bindValue(':mother_name', $input['mother_name']);
    $stmt->bindValue(':mother_aadhar', $input['mother_aadhar']);
    $stmt->bindValue(':mother_phone', $input['mother_phone']);
    $stmt->bindValue(':mother_edu', $input['mother_edu']);
    $stmt->bindValue(':mother_occ', $input['mother_occ']);
    $stmt->bindValue(':mother_desig', $input['mother_desig']);
    $stmt->bindValue(':mother_org', $input['mother_org']);
    $stmt->bindValue(':mother_org_addr', $input['mother_org_addr']);

    $stmt->bindValue(':single_parent', $input['single_parent']);
    $stmt->bindValue(':sibling_name', $input['sibling_name']);
    $stmt->bindValue(':sibling_class', $input['sibling_class']);
    $stmt->bindValue(':medical_need', $input['medical_need']);
    $stmt->bindValue(':medical_details', $input['medical_details']);
    $stmt->bindValue(':transfer_info', $input['transfer_info']);
    $stmt->bindValue(':transport_req', $input['transport_req']);
    $stmt->bindValue(':category', $input['category']);

    $stmt->bindValue(':declaration', $input['declaration'], PDO::PARAM_INT);
    $stmt->bindValue(':photo_child_path', $input['photo_child_path']);
    $stmt->bindValue(':photo_father_path', $input['photo_father_path']);
    $stmt->bindValue(':photo_mother_path', $input['photo_mother_path']);
    $stmt->bindValue(':sign_mother_path', $input['sign_mother_path']);
    $stmt->bindValue(':sign_father_path', $input['sign_father_path']);

    $stmt->execute();
    $insertId = $pdo->lastInsertId();

    $pdo->commit();

    // success
    respond(true, 'Application received and saved (ID: ' . $insertId . ')', ['id' => $insertId]);

} catch (PDOException $ex) {
    if (isset($pdo) && $pdo->inTransaction()) $pdo->rollBack();
    error_log('DB error: ' . $ex->getMessage());
    respond(false, 'Server error: database operation failed.');
} catch (Exception $ex) {
    if (isset($pdo) && $pdo->inTransaction()) $pdo->rollBack();
    error_log('General error: ' . $ex->getMessage());
    respond(false, 'Server error.');
}
